(in-package "CL-USER")

; This file contains the functions for type checking and type
; fixing. As a side effect, type checking puts the formula into the
; dag form that will be used for simplification. Each node in the dag
; has a slot for the type, which type checking also provides.

; There are two kinds of types: definite, and abstract. Definite types
; are positive integers, and correspond to the size of the bit vector
; returned by the formula. Abstract types are used when we do not know
; the type for sure. For example, formulas can contain integers such
; as 7. It is unclear how big a bit-vector will be used to represent
; 7. In these cases, the type is (>= . n) where n is the minimum
; number of bits needed to represent the formula. For this
; calculation, integers are assumed to be in signed
; representation. Hence, 7 has type (>= . 4).

; The the fixing functions replace integers with bit-vectors, using
; the context to figure out the definite type to give the vector. For
; example, if I have the formula (= x 7), and x is a 6 bit bit-vector,
; fix-ints will replace 7 with (const 0 0 0 1 1 1). This, in turn
; eliminates the abstract types, leaving us with only definite types.

;; (load "syntax-check")

;returns the bits of the variable varname in typing environment e.
; this function will now return the no. of words
(defun get-var-bits (varname e)
  (second (assoc varname e)))

;returns the word-size of the variable varname in typing environment e.
;changed by roma on Nov 18,05 to get the word-size from variable defn.
(defun get-var-word-size (varname e)
  (if (third (assoc varname e))
      (third (assoc varname e))
    1))

;given a bitsize, it returns the minimum bitsize. That is, if tp is
;abstract, it returns the minimum possible bitsize.
(defun minbitsize (tp)
  (if (atom tp)
      tp
    (cdr tp)))

(defun most-restrictive-type1 (mv1 mv2 nt)
  (cond ((endp mv1) (when (endp mv2) (cons 'mv (reverse nt))))
	((endp mv2) nil)
	(t
	 (case (compare-types (car mv1) (car mv2))
	   (<= (most-restrictive-type1 (cdr mv1) (cdr mv2) (cons (car mv1) nt)))
	   (=  (most-restrictive-type1 (cdr mv1) (cdr mv2) (cons (car mv1) nt)))
	   (>= (most-restrictive-type1 (cdr mv1) (cdr mv2) (cons (car mv2) nt)))
	   (otherwise nil)))))

(defun most-restrictive-type (t1 t2)
  (cond ((eq (car t1) 'mv)
	 (when (eq (car t2) 'mv) (most-restrictive-type1 (cdr t1) (cdr t2) nil)))
	((eq (car t2) 'mv) nil)
	(t
	 (case (compare-types t1 t2)
	   (<= t1)
	   (= t1)
	   (>= t2)
	   (otherwise nil)))))
  
(defun compare-types1 (mv1 mv2 nt)
  (cond ((endp mv1) (when (endp mv2) (reverse nt)))
	((endp mv2) nil)
	(t
	 (let ((ct (compare-types (car mv1) (car mv2))))
	   (when ct
	     (compare-types1 (cdr mv1) (cdr mv2) (cons ct nt)))))))

;; returns <= if t1 is a subtype of t2
;; returns >= if t2 is a subtype of t1
;; returns =  if t1 = t2
;; returns nil if t1 and t2 are incompatible
(defun compare-types (t1 t2)
  (declare (type list t1) (type list t2))
  (let ((tt1 (car t1))
	(tt2 (car t2)))
    (cond ((eq tt1 'mem) (when (equal t1 t2) '=))
	  ((eq tt1 'mv)
	   (when (eq tt2 'mv) (compare-types1 (cdr t1) (cdr t2) nil)))
	  ((or (eq tt2 'mem)
	       (eq tt2 'mv))
	   nil)
	  ((eq tt1 'bv)
	   (if (eq tt2 'bv)
	       (when (= (second t1) (second t2)) '=)
	     (when (<= (second t2) (second t1)) '<=)))
	  (t
	   (if (eq tt2 'bv)
	       (when (<= (second t1) (second t2)) '>=)
	     (cond ((= (second t1) (second t2)) '=)
		   ((< (second t1) (second t2)) '>=)
		   (t '<=)))))))

(defun subtypeof1 (lst)
  (or (endp lst)
      (and (or (eq (car lst) '<=) (eq (car lst) '=))
	   (subtypeof1 (cdr lst)))))

(defun subtypeof (t1 t2)
  (let ((ct (compare-types t1 t2)))
    (or (eq ct '<=)
	(eq ct '=)
	(and (consp ct)
	     (subtypeof1 ct)))))

(defun compat-lst1 (lst tl tcf oitem tp nlst)
  (if (endp lst) 
      (values (if tl (check-ambiguity tp) tp) (reverse nlst))
    (let ((f (funcall tcf (car lst))))
      (when f 
	(let ((mrt (most-restrictive-type (formula-type f) tp)))
	  (cond ((not mrt)
		 (format t "~&Incompatible types of arguments: ~a:~a, ~a:~a"
			 oitem tp (car lst) (formula-type f)))
		((eq mrt tp)
		 (compat-lst1 (cdr lst) tl tcf oitem tp (cons f nlst)))
		(t
		 (compat-lst1 (cdr lst) tl tcf (car lst) mrt (cons f nlst)))))))))

;; 	(let ((ct (compare-types (formula-type f) tp)))
;; 	  (case ct
;; 	    (<=  (compat-lst1 (cdr lst) tl tcf oitem tp (cons f nlst)))
;; 	    (=   (compat-lst1 (cdr lst) tl tcf oitem tp (cons f nlst)))
;; 	    (>=  (compat-lst1 (cdr lst) tl tcf (car lst) (formula-type f) (cons f nlst)))
;; 	    (otherwise (format t "~&Incompatible types of arguments: ~a:~a, ~a:~a"
;; 			 oitem tp (car lst) (formula-type f)))))))))

(defun compat-lst (lst tl tcf)
  (let ((f (funcall tcf (car lst))))
    (when f
      (compat-lst1 (cdr lst) tl tcf (car lst) (formula-type f) (list f)))))

(defmacro with-formula (v cf &rest body)
  `(let ((,v ,cf))
     (when ,v . ,body)))

;; (defun with-formulas-fn (bindings body)
;;   (if (endp bindings)
;;       body
;;     `(with-formula ,(caar bindings) ,(cadar bindings)
;; 		   ,(with-formulas-fn (caar bindings) (cadar bindings)))))

(defmacro with-formulas (bindings &rest body)
  (cond ((endp bindings)
	 (cons 'progn body))
	((endp (cdr bindings))
	 `(with-formula ,(caar bindings) ,(cadar bindings) ,@body))
	(t `(with-formula ,(caar bindings) ,(cadar bindings)
			  (with-formulas ,(cdr bindings) ,@body)))))

(defun r-type-check (form types pform db e d)
  (with-formulas
   ((f (type-check form db e d)))
   (if (member (car (formula-type f)) types :test 'eq)
       f
     (format t "~&~a has illegal type ~a in form ~a."
	     form
	     (formula-type f)
	     pform))))


(defun r-tl-type-check (form types pform db e d)
  (with-formulas
   ((f (r-type-check form types pform db e d)))
   (setf (formula-type f)
	 (check-ambiguity (formula-type f)))
   f))

;;;;;;;;;;;;;;;;;;;;;;;;

;; ;returns whether t1 and t2 are compatible bitsizes.
;; (defun compatible-bitsizes (t1 bs1 t2 bs2)
;;   (cond ((or (eq t1 'mem) (eq t2 'mem)) nil)
;; 	((eq t1 'bv)
;; 	 (if (eq t2 'bv)
;; 	     (= bs1 bs2)
;; 	   (<= bs2 bs1)))
;; 	(t
;; 	 (if (eq t2 'bv)
;; 	     (<= bs1 bs2)
;; 	   nil))))

;;   (cond ((and (atom t1)
;; 	      (atom t2))
;; 	 (= t1 t2))         ;if they are both definite, they must be equal
;; 	((atom t1)
;; 	 (<= (cdr t2) t1))  ;if only one is definite, it must be greater than the minbitsize of the other.
;; 	((atom t2)
;; 	 (<= (cdr t1) t2))
;; 	(t t)))             ;two abstract bitsizes are always compatible.

;; ;t1 is a subtype of t2 if they are compatible and the minbitsize of t1 is at most that of t2.
;; (defun subtypeof (t1 bs1 t2 bs2)
;;   (and (compatible-bitsizes t1 bs1 t2 bs2 )
;;        (<= bs1 bs2)))

;; (defun memsubtypeof (t1 bs1 ws1 t2 bs2 ws2)
;;   (cond ((eq t1 'mem) (and (eq t2 'mem)
;; 			   (= bs1 bs2)
;; 			   (= ws1 ws2)))
;; 	((eq t2 'mem) nil)
;; 	(t (subtypeof t1 bs1 t2 bs2))))

;; ;chooses the most restrictive of the two bitsizes.
;; (defun pick-bitsize (t1 bs1 t2 bs2)
;;   (cond ((and (eq t1 'int) (eq t2 'int))
;; 	 (values 'int (max bs1 bs2)))
;; 	((eq t1 'int) (values 'bv bs2))
;; 	((eq t2 'int) (values 'bv bs1))
;; 	(t (values 'bv bs1))))
;;   (if (or (and (consp t1)
;; 	       (atom t2))
;; 	  (and (consp t1)
;; 	       (consp t2)
;; 	       (< (cdr t1) (cdr t2))))
;;       t2
;;     t1))

;; ; added by roma to pick appropriate wordsize
;; ; if the wordsizes are different, but no. of bits are the same, wordsize of result is 1
;; (defun pick-wordsize (w1 w2)
;;   (cond ((= w1 w2) w1)
;; 	(t 
;; 	 (format t "~& The wordsizes of the arguments don't match, wordsize of the result is 1")
;; 	 1)))

;does the same as min-bitsize, but gives a warning if the bitsize is
;ambiguous. This is used for top level expressions, where there is no
;context to infer the definite bitsize of an ambiguously bitsized statement.
(defun check-ambiguity (tp)
;;  (declare (ignore form))
  (cond ((eq (car tp) 'int)
	 (format t "~&Warning: top-level expression is of ambiguous bitsize. Will use smallest possible bitsize.")
	 (cons 'bv (rest tp)))
	(t tp)))
;;   (if (atom tp)
;;       tp
;;     (progn
;;       (format t "~&Warning: top-level expression is of ambiguous bitsize. Will use smallest possible bitsize.")
;; ;;      (format t "~&bitsize: ~a~&form: ~a." tp form)
;;       (cdr tp))))

;type-checks a list of forms, making sure they are all of compatible
;type. 
;a is the list being checked 
;tl is a boolean that indicates if these are top-level expressions
; (whether we should check for ambiguity) 
;db lets us know which bits of varibles are defined. This is important
; in the case of locals, where users can define variables bit by bit.
;e is the typing environment
;d is the description
;; (defun compatible-typelist1 (a db e d tp bits ca)
;;   (if (endp a)
;;       (values tp bits nil)
;;     (let ((f (type-check (car a) db e d)))
;;       (cond ((not f)
;; 	     (values nil nil nil))
;; 	    ((eq (formula-type f) 'mem)
;; 	     (values (format t "~Type Error: Illegal use of memory: ~a" (car a)) nil nil))
;; 	    (t
;; 	     (let ((fbits (formula-bits f))
;; 		   (ftp   (formula-type f)))
;; 	       (if (compatible-bitsizes tp bits ftp fbits)
;; 		   (multiple-value-bind
;; 		     (ntp nbits)
;; 		     (pick-bitsize tp bits ftp fbits)
;; 		     (multiple-value-bind
;; 		       (rtp rbits rforms)
;; 		       (compatible-typelist1 (cdr a) db e d ntp nbits
;; 					     (if (and (eq ntp tp)
;; 						      (= nbits bits))
;; 						 ca 
;; 					       (car a)))
;; 		       (if rtp 
;; 			   (values rtp rbits (cons f rforms))
;; 			 (values nil nil nil))))
;; 		 (values (format t "~&Bits Error: Bits (type) mismatch: ~&~a:~a~&~a:~a"
;; 				 ca bits (car a) fbits)
;; 			 nil nil))))))))

;; (defun compatible-typelist (a tl db e d)
;;   (multiple-value-bind
;;     (tp bits formlist)
;;     (compatible-typelist1 a db e d 'int 0 nil)
;;     (cond ((not tp) (values nil nil nil))
;; 	  (tl       (values (check-ambiguity tp) bits formlist))
;; 	  (t        (values tp bits formlist)))))

;; (defun mem-compatible-typelist1 (a db e d bits ws ca)
;;   (if (endp a)
;;       (values nil nil)
;;     (let ((f (type-check (car a) db e d)))
;;       (when f
;; 	(if (and (eq (formula-type f) 'mem)
;; 		 (= (formula-bits f) bits)
;; 		 (= (formula-wordsize f) ws))
;; 	    (multiple-value-bind 
;; 	      (forms errp)
;; 	      (mem-compatible-typelist1 (cdr a) db e d bits ws ca)
;; 	      (if errp (values nil t) (values (cons f forms) nil)))
;; 	  (values nil t))))))

;; (defun mem-compatible-typelist (a tl db e d)
;;   (let ((f (type-check (car a) db e d)))
;;     (cond ((not f) nil)
;; 	  ((eq (formula-type f) 'mem)
;; 	   (multiple-value-bind
;; 	     (forms errp)
;; 	     (mem-compatible-typelist1 (cdr a) db e d
;; 				       (formula-bits f) 
;; 				       (formula-wordsize f)
;; 				       (car a))
;; 	     (if errp
;; 		 (values nil nil nil nil)
;; 	       (values 'mem (formula-bits f) (formula-wordsize f) forms))))	  
;; 	  (t
;; 	   (multiple-value-bind
;; 	     (tp bits formlist)
;; 	     (compatible-typelist1 (cdr a) db e d (formula-type f) (formula-bits f) (car a))
;; 	     (cond ((not tp) (values nil nil nil nil))
;; 		   (tl       (values (check-ambiguity tp) bits 1 (cons f formlist)))
;; 		   (t        (values tp bits 1 (cons f formlist)))))))))

;; (defun compatible-typelist (a tl db e d)
;; ;  (format t "~&compatible-typelist. a: ~a" a)
;;   (let ((tp 'int)
;; 	(tp2 nil)
;; 	(bits 1)
;; 	(bits2 0)
;; 	(current-arg nil)
;; 	(formlist nil))
;;     (dolist (x a (values (if tl
;; 			     (check-ambiguity bits x)
;; 			   bits) formlist))
;;       (let ((f (type-check x db e d)))
;; 	(if f
;; 	    (progn
;; 	      (setf formlist (append formlist (list f)))
;; 	      (setf bits2 (formula-bits f))
;; 	      (if (compatible-bitsizes bits bits2)
;; 		  (progn 
;; 		    (setf bits (pick-bitsize bits bits2))
;; 		    (when (equal bits bits2) (setf current-arg x)))
;; 		(return (values (format t "~&Bits Error: Bits (type) mismatch: ~&~a:~a~&~a:~a"
;; 					current-arg bits x bits2)
;; 				nil nil))))
;; 	(return (values nil nil nil)))))))
	
;a top-level type-check. it checks for ambiguity.
;form is the form being checked.
;db lets us know which bits of varibles are defined. This is important
; in the case of locals, where users can define variables bit by bit.
;e is the typing environment
;d is the description
(defun tl-type-check (form db e d)
  (let ((tc (type-check form db e d)))
    (when tc
      (setf (formula-type tc)
	    (check-ambiguity (formula-type tc)))
      tc)))
;    (when (and tc
;	       (check-ambiguity (formula-type tc) form))
;      tc)))

;certain formulas (such as tests for ifs) are required to be of type
;1. this funciton type checks form and verifies that it is of type 1.
;form is the form being checked.
;db lets us know which bits of varibles are defined. This is important
; in the case of locals, where users can define variables bit by bit.
;e is the typing environment
;d is the description
(defun type-check-1 (form db e d)
  (let ((tc (tl-type-check form db e d)))
    (when tc
      (if (equal (formula-type tc) '(bv 1))
	  tc
	(format t "~&Bits(type) error: expecting a formula of bits(type) 1. 
You wrote: ~a" form)))))

;returns a list of the integers between r1 and r2
(defun list-range (r1 r2)
  (do ((x r2)
       (lst nil))
      ((<= x (1- r1)) lst)
    (setf lst (cons x lst))
    (setf x (1- x))))

;define which bits are set in our defbits list, lst.
(defun set-bits (lst var bits)
  (cond ((endp lst) nil)
	 ;; (list (cons var bits))) ;DARON: just changed this.
	((eq (caar lst) var)
	 (cons (cons var bits)
	       (cdr lst)))
	(t (cons (car lst) (set-bits (cdr lst) var bits)))))

;merges 2 sorted lists of bits into 1 sorted list of bits. used to add
;bits to the list of those defined for a variable.
(defun merge-bits (lst1 lst2)
  (cond ((endp lst1) lst2)
	((endp lst2) lst1)
	(t
	 (let ((i1 (car lst1))
	       (i2 (car lst2)))
	   (cond ((< i1 i2) (cons i1 (merge-bits (cdr lst1) lst2)))
		 ((> i1 i2) (cons i2 (merge-bits lst1 (cdr lst2))))
		 (t         (cons i1 (merge-bits (cdr lst1) (cdr lst2)))))))))

;modifies the list of bits defined for a given variable in the defbits list lst.
(defun add-bits (lst var bits)
  (cond ((endp lst) (list (cons var bits)))
	((eq (caar lst) var)
	 (cons (cons var (merge-bits bits (cdar lst)))
	       (cdr lst)))
	(t (cons (car lst) (add-bits (cdr lst) var bits)))))

;returns whether bit bit of variable var is defined according to defbits.
(defun bit-defined? (defbits var bit)
  (let ((v (assoc var defbits)))
    (or (not v)
	(member bit (cdr v)))))

;returns whether a range of bits is defined for variable var according
;to defbits.
(defun bit-range-defined? (defbits var b1 b2)
  (let ((v (assoc var defbits)))
    (if (not v)
	t
      (do ((v (cdr v))
	   (c b1))
	((<= c (1+ b2)) t) ;DARON: just changed
;	(format t "~&  c: ~a v: ~a" c v)
	(cond ((endp v) (return nil))
	      ((< (car v) c)
	       (setf v (cdr v)))
	      ((= (car v) c)
	       (setf v (cdr v))
	       (setf c (1+ c)))
	      (t (return nil)))))))

;returns whether any of the bits between b1 and b2 are defined for
;variable var according to defbits.
(defun any-bit-range-defined? (defbits var b1 b2)
  (let ((v (assoc var defbits)))
    (if (not v)
	t
      (do ((v (cdr v))
	   (c b1))
	  ((<= c (1+ b2)) nil) ;DARON: just changed
	(cond ((endp v) (return nil))
	      ((< (car v) c)
	       (setf v (cdr v)))
	      ((= (car v) c)
	       (return t))
	      (t (setf v (cdr v))
		 (1+ c)))))))

;returns whether all of variable var is defined according to defbits.
(defun var-defined? (defbits var e)
  (let ((vtp (third (assoc var e))))
    ;; (format t "~&var-defined?: (assoc var e) = ~a" (assoc var e))
    (bit-range-defined? defbits var 0 (1- vtp))))

;returns whether any bit of variable var is defined according to defbits.
(defun any-var-defined? (defbits var e)
  (let ((vtp (get-var-bits var e)))
    (any-bit-range-defined? defbits var 0 (1- vtp))))

;removes a var from defbits.
(defun remove-from-defbits (defbits var)
  (cond ((endp defbits)
	 nil)
	((eq (caar defbits) var)
	 (remove-from-defbits (cdr defbits) var))
	(t
	 (cons (car defbits)
	       (remove-from-defbits (cdr defbits) var)))))

(defun lbtc-cat (v vars defbits e b)
  (if (endp v)
      (values nil e defbits 0 nil)
    (let ((x (car v)))
      (cond ((and (atom x)
		  (member x vars)) ;; defining all of a var declared to be defined by bits.
	     (let* ((def (assoc x e))
		    (tp (cdr def)))
	       (cond ((any-var-defined? defbits x e)
		      (values (format t "~&Attempt to redefine ~a: ~a" x b) nil nil 0 t))
		     (t
		      (multiple-value-bind
			(vforms ne ndb bsum errp)
			(lbtc-cat (cdr v) 
				  vars
				  (set-bits defbits x (list-range 0 (1- (get-var-bits x e))))
				  e b)
			(if errp
			    (values nil nil nil 0 t)
			  (values (cons (make-formula :fn 'var-cat-binding
						      :type tp
						      :args (list x))
					vforms)
				  ne
				  ndb
				  (+ (second tp) bsum)
				  nil)))))))
	    ((atom x)
	     (multiple-value-bind
	       (vforms ne ndb bsum errp)
	       (lbtc-cat (cdr v) vars (remove-from-defbits defbits x) (cons (list x 'bv 1) e) b)
	       (if errp
		   (values nil nil nil 0 t)
		 (values (cons (make-formula :fn 'new-cat-binding
					     :args (list x))
			       vforms)
			 ne
			 ndb
			 (+ 1 bsum)
			 nil))))
	    ((and (= (length x) 2)
		  (member (first x) vars :test 'eq)) ;; setting a bit of a var
	     (let ((bits (third (assoc (first x) e :test 'eq)))
		   (bit (second x)))
	       (cond ((>= bit bits)
		      (values (format t "~&Type error: bit index out of bounds in binding: ~a" x)
			      nil nil 0 t))
		     ((bit-defined? defbits (car x) bit)
		      (values (format t "~&Attempt to redefine ~a: ~a" x b)
			      nil nil 0 t))
		     (t
		      (multiple-value-bind
			(vforms ne ndb bsum errp)
			(lbtc-cat (cdr v) vars (add-bits defbits (car x) (list bit)) e b)
			(if errp
			    (values nil nil nil 0 t)
			  (values (cons (make-formula :fn 'bit-cat-binding
						      :args x)
					vforms)
				  ne
				  ndb
				  (+ 1 bsum)
				  nil)))))))
	    ((= (length x) 2) ;; new variable, declaring size.
	     (let ((bits (second x)))
	       (multiple-value-bind
		 (vforms ne ndb bsum errp)
		 (lbtc-cat (cdr v)
			   vars
			   (remove-from-defbits defbits x)
			   (acons (first x) `(bv ,bits) e) b)
		 (if errp
		     (values nil nil nil 0 t)
		   (values (cons (make-formula :fn 'new-cat-binding
					       :type `(bv ,bits)
					       :args (list (car x)))
				 vforms)
			   ne
			   ndb
			   (+ bsum bits)
			   nil)))))
	    ((and (= (length x) 3)
		  (member (first x) vars))
	     (let* ((var (car x))
		    (b1 (second x))
		    (b2 (third x))
		    (bits (+ (- b2 b1) 1))
		    (def (assoc var e))
		    (vbits (second def)))
	       (cond ((<= b2 b1)
		      (values (format t 
				      "~&Type error: second bit number in bit range must be greater than the first: ~a"
				      x)
			      nil nil 0 t))
		     ((<= vbits b2)
		      (values (format t "~&Type error: bit index out of bounds in binding: ~a" x)
			      nil nil 0 t))
		     ((any-bit-range-defined? defbits (car x) b1 b2)
		      (values (format t "~&Attempt to redefine ~a: ~a" x b)
			      nil nil 0 t))
		     (t
		      (multiple-value-bind
			(vforms ne ndb bsum errp)
			(lbtc-cat (cdr v)
				  vars
				  (add-bits defbits (car x) (list-range b1 b2))
				  e b)
			(if errp
			    (values nil nil nil 0 t)
			  (values (cons (make-formula :fn 'bits-cat-binding
						      :type `(bv ,bits)
						      :args x)
					vforms)
				  ne
				  ndb
				  (+ bsum bits)
				  nil)))))))
	    (t ;; length = 3, not in vars.
	     (format t "~&Invalid binding: ~a" b))))))

(defun lbtc-binding (b etp exp mv? vars defbits e)
  (let ((v (car b))
	(len (length b)))
    (cond ((= len 2) ;; set a var to a bv expression, providing a definite size.
	   (let* ((bts (second b))
		  (tp `(bv ,bts)))
	     (cond  ((member v vars)
		     (values (format t "~&Attempt to redeclare variable: ~a" b) nil nil))
		    ((subtypeof tp etp)
		     (values (if mv?
				 (make-formula :fn 'new-mv-binding
					       :type tp
					       :args (list v))
			       (make-formula :fn 'new-binding
					     :type tp
					     :args (list v exp)))
			     nil
			     (acons v tp e)))
		    (t (values (format t "~&Type error: Incompatible types in binding: ~a" b)
			       nil
			       nil)))))
					; changed by roma on Nov 18,05 to take care of addition of word-size in local bindings
	  ((= len 3) ;; setting a var to a memory
	   (let* ((nw (second b))
		  (ws (third b))
		  (tp `(mem ,nw ,ws)))
	     (cond ((member v vars) 
		    (values (format t "~&Attempt to redeclare variable: ~a" b) nil nil))
		   ((subtypeof tp etp)
		    (values (if mv?
				(make-formula :fn 'new-mv-binding
					      :type tp
					      :args (list v))
			      (make-formula :fn 'new-binding
					    :type tp
					    :args (list v exp)))
			    (remove-from-defbits defbits v)
			    (acons v tp e)))
		   (t (values (format t "~&Type error: Incompatible types in binding: ~a" b)
			      nil nil)))))
	  ;; len = 1 from here on
	  ((and (atom v)
		(member v vars))
	   (let* ((tp (cdr (assoc v e))))
	     (cond  ((any-var-defined? defbits v e)
		     (values (format t "~&Attempt to redefine ~a: ~a" v b)
			     nil nil))
		    ((subtypeof tp etp)
		     (values (if mv?
				 (make-formula :fn 'var-mv-binding
					       :type tp
					       :args (list v))
			       (make-formula :fn 'var-binding
					     :type tp
					     :args (list v exp)))
			     (set-bits defbits v (list-range 0 (1- (get-var-bits v e))))
			     e))
		    (t (values (format t "~&Type error: Incompatible types in binding: ~a" b)
			       nil nil)))))
	  ((atom v)
	   (values (if mv?
		       (make-formula :fn 'new-mv-binding
				     :type (check-ambiguity etp)
				     :args (list v))
		     (make-formula :fn 'new-binding
				   :type (check-ambiguity etp)
				   :args (list v exp)))
		   (remove-from-defbits defbits v)
		   (acons v etp e)))
	  ((and (endp (cddr v))
		(integerp (second v)))
	   (let ((bit (second v))
		 (vbts (get-var-bits (car v) e)))
	     (cond  ((bit-defined? defbits (car v) bit)
		     (values (format t "~&Attempt to redefine ~a: ~a" v b)) nil nil)
		    ((not (< bit vbts))
		     (values (format t "~&Bits(type) error: bit index out of bounds in binding: ~a" b)
			     nil nil))
		    ((subtypeof etp '(bv 1))
		     (values (if mv?
				 (make-formula :fn 'bit-mv-binding
					       :args (list v exp))
			       (make-formula :fn 'bit-binding
					     :args (list v exp)))
			     (add-bits defbits (car v) (list bit))
			     e))
		    (t (values (format t "~&Type error: Incompatible types in binding: ~a" b)
			       nil nil)))))
	  ((and (endp (cdddr v))
		(integerp (second v)))
	   (let* ((b1 (nth 1 v))
		  (b2 (nth 2 v))
		  (bts (+ (- b2 b1) 1))
		  (vbts (get-var-bits (car v) e)))
	     (cond ((not (< b1 b2))
		    (values (format t 
				    "~&Type error: second bit number in bit range must be greater than the first: ~a"
				    b)
			    nil nil))
		   ((not (< b2 vbts))
		    (values (format t "~&Bits(type) error: bit index out of bounds in binding: ~a" b)
			    nil nil))
		   ((any-bit-range-defined? defbits (car v) b1 b2)
		    (values (format t "~&Attempt to redefine ~a: ~a" v b)
			    nil nil))
		   ((subtypeof `(bv ,bts) etp)
		    (values (if mv?
				(make-formula :fn 'bits-mv-binding
					      :args (list v))
			      (make-formula :fn 'bits-binding
					    :args (list v exp)))
			    (add-bits defbits (car v) (list-range b1 b2))
			    e))
		   (t (values (format t "~&Type error: Incompatible types in binding: ~a" b)
			      nil nil)))))
	  (t
	   (cond ((eq (car etp) 'mv)
		  (multiple-value-bind
		    (vforms btp ndb ne)
		    (lbtc-mv v (cdr etp) vars defbits e nil nil)
		    (cond ((not vforms) (values nil nil nil))
			  (mv? (values (make-formula :fn 'mv-mv-binding
						     :type btp
						     :args (list vforms))
				       ndb
				       ne))
			  (t   (values (make-formula :fn 'mv-binding
						     :type btp
						     :args (list vforms exp))
				       ndb
				       ne)))))
		 (t
		  (multiple-value-bind
		    (vforms ne ndb bsum errp)
		    (lbtc-cat v vars defbits e b)
		    (if errp
			(values nil nil nil)
		      (cond ((subtypeof etp
					`(bv ,bsum))
			     (values (if mv?
					 (make-formula :fn 'cat-mv-binding
						       :type `(bv ,bsum)
						       :args (list vforms))
				       (make-formula :fn 'cat-binding
						     :type `(bv ,bsum)
						     :args (list vforms exp)))
				     ndb
				     ne))
			    (t 
			     (values (format t "~&Type error: Incompatible types in binding: ~a" b)
				     nil nil)))))))))))

(defun lbtc-mv (v etps vars defbits e tp bforms)
  (if (endp v)
      (values (reverse bforms) (cons 'mv (reverse tp)) defbits e)
    (multiple-value-bind
      (bform ndb ne)
      (lbtc-binding (if (atom (car v)) (list (car v)) (car v))
		    (car etps) nil t vars defbits e)
      (if bform
	  (lbtc-mv (cdr v) (cdr etps) vars ndb ne
		   (cons (formula-type bform) tp) (cons bform bforms))
	(values nil nil nil)))))

(defun lbtc1 (bindings vars defbits e d bforms)
  (if (endp bindings)
      (values defbits e (reverse bforms))
    (let ((tc (type-check (car (last (car bindings))) defbits e d)))
      (if tc
	  (multiple-value-bind
	    (bform ndb ne)
	    (lbtc-binding (butlast (car bindings))
			  (formula-type tc)
			  tc nil vars defbits e)
	    (if bform
		(lbtc1 (cdr bindings) vars ndb ne d (cons bform bforms))
	      (values nil nil nil)))
	(values nil nil nil)))))

;type-checks local bindings.
;bindings is the list of bindings to be checked.
;vars is the list of variables to be bound bit-by-bit.
;e is the typing environment.
;d is the desc.
(defun local-binding-type-check (bindings vars defbits e d)
;;  (format t "~&local-binding-type-check: defbits=~a" defbits)
  (lbtc1 bindings vars defbits e d nil))

;; 		     (let* ((tsum 0)
;; 			    (bforms2 nil)
;; 			    (cont (dolist (x v t)
;; 				    ;(format t "~&x: ~a" x)
;; 				    (cond ((and (atom x)
;; 						(member x vars))
;; 					   (let* ((def (assoc x e))
;; 						  (bits (second def))
;; 						  (ws (third def)))
;; 					     (cond ((any-var-defined? defbits x e)
;; 						    (return (format t "~&Attempt to redefine ~a: ~a" x b)))
;; 						   (t
;; 						    (setf bforms2 (app-item bforms2
;; 									    (make-formula :fn 'var-mv-binding
;; 											  :bits bits
;; 											  :wordsize ws
;; 											  :args (list x))))
;; 						    (setf defbits 
;; 							  (set-bits defbits x (list-range 0 (1- (get-var-bits x e)))))
;; 						    (setf tsum (+ tsum bits))))))
;; 					  ((atom x)
;; 					   (setf bforms2 (app-item bforms2 
;; 								   (make-formula :fn 'new-mv-binding
;; 										 :args (list x))))
;; 					   (setf tsum (+ tsum 1))
;; 					   (setf defbits (remove-from-defbits defbits x))
;; 					   (setf e (cons (list x 1 1) e)))
;; 					  ((and (= (length x) 2)
;; 						(member (car x) vars))
;; 					   (let ((bits (get-var-bits (car x) e))
;; 						 (bit (second x)))
;; 					     (cond ((>= bit bits)
;; 						    (return (format t "~&Type error: bit index out of bounds in binding: ~a" x)))
;; 						   ((bit-defined? defbits (car x) bit)
;; 						    (return (format t "~&Attempt to redefine ~a: ~a" x b)))
;; 						   (t
;; 						    (setf bforms2 (app-item bforms2
;; 									    (make-formula :fn 'bit-mv-binding
;; 											  :args x)))
;; 						    (setf defbits (add-bits defbits (car x) (list bit)))
;; 						    (setf tsum (+ tsum 1))))))
;; 					  ((= (length x) 2)
;; 					   (let ((bits (second x)))
;; 					     (setf bforms2 (app-item bforms2
;; 								     (make-formula :fn 'new-mv-binding
;; 										   :bits bits
;; 										   :args (list (car x)))))
;; 					     (setf tsum (+ tsum bits))
;; 					     (setf defbits (remove-from-defbits defbits v))
;; 					     (setf e (cons (list (first x) (second x) 1) e))))
;; 					  ((= (length x) 3)
;; 					   (let* ((ws (third x))
;; 						  (bits (* ws (second x))))
;; 					     (setf bforms2 (app-item bforms2
;; 								     (make-formula :fn 'new-mv-binding
;; 										   :bits bits
;; 										   :wordsize ws
;; 										   :args (list (car x)))))
;; 					     (setf tsum (+ tsum bits))
;; 					     (setf defbits (remove-from-defbits defbits v))
;; 					     (setf e (cons (list (first x) bits ws) e))))

;; 					  (t
;; 					   (let* ((var (car x))
;; 						  (b1 (second x))
;; 						  (b2 (third x))
;; 						  (bits (+ (- b2 b1) 1))
;; 						  (def (assoc var e))
;; 						  (vbits (second def)))
;; 					     (cond ((<= b2 b1)
;; 						    (return (format t 
;; 								    "~&Type error: second bit number in bit range must be greater than the first: ~a"
;; 								    x)))
;; 						   ((<= vbits b2)
;; 						    (return (format t "~&Type error: bit index out of bounds in binding: ~a" x)))
;; 						   ((any-bit-range-defined? defbits (car x) b1 b2)
;; 						    (return (format t "~&Attempt to redefine ~a: ~a" x b)))
;; 						   (t 
;; 						    (setf bforms2 (app-item bforms2
;; 									    (make-formula :fn 'bits-mv-binding
;; 											  :bits bits
;; 											  :args x)))
;; 						    (setf defbits (add-bits defbits (car x) (list-range b1 b2)))
;; 						    (setf tsum (+ tsum bits))))))))))
;; 		       (if cont
;; 			   (if (subtypeof (formula-bits tc)
;; 					  tsum)
;; 			       (setf bforms (app-item bforms
;; 						      (make-formula :fn 'mv-binding
;; 								    :bits tsum
;; 								    :args (list bforms2 tc))))
;; 			     (return (format t "~&Type error: Incompatible types in binding: ~a" b)))
;; 			 (return nil)))
;; 		   (return nil)))))))))
				  

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; added by roma on 9 march - adding a function
; string-list to convert string to list - used in binary
; changed to implement Verilog representation - 18 May

; this function converts to binary list of 4 bits exact
(defun bits-4 (blist)
  (if (and (consp blist) 
	   (true-listp blist))
      (if (<= (length blist) 3)
	  (bits-4 (cons '0 blist))
	(if (= (length blist) 4)
	    blist
	  '(0 0 0 0)))
    '(0 0 0 0)))

; this function converts a hex-char in string to given bit vector
(defun hex-char-bv (x)
  (let ((ch (char-code x)))
    (if (and (>= ch 48)
	     (<= ch 57))
	(n-ubv (- ch 48))
      (if (and (>= ch 97)
	       (<= ch 102))
	  (n-ubv (- ch 87))
	nil))))

; this function converts hex list to binary
(defun hex-list-binary (form)
  (if (consp form)
      (let* ((x (car form))
	     (y (hex-char-bv x)))
	(append (bits-4 y) (hex-list-binary (cdr form))))
    nil))  
		    
(defun hex-size (size hex)
  (if (< (length hex) size)
      (hex-size size (append '(#\0) hex))
    hex))
 
(defun hex-size-error (size hex)
  (if (> (length hex) size)
      (format t "Error: Given size is less than required representation size")
    (hex-size size hex)))


;(* main function *)
;;;;;;;;;;;;;;;;;;;;
(defun hex-binary (hexval)
  (loop for hchar in (cddr (coerce hexval 'list))
	for tuple = (case hchar
		      (#\0 '(0 0 0 0))
		      (#\1 '(0 0 0 1))
		      (#\2 '(0 0 1 0))
		      (#\3 '(0 0 1 1))
		      (#\4 '(0 1 0 0))
		      (#\5 '(0 1 0 1))
		      (#\6 '(0 1 1 0))
		      (#\7 '(0 1 1 1))
		      (#\8 '(1 0 0 0))
		      (#\9 '(1 0 0 1))
		      (#\a '(1 0 1 0))
		      (#\b '(1 0 1 1))
		      (#\c '(1 1 0 0))
		      (#\d '(1 1 0 1))
		      (#\e '(1 1 1 0))
		      (#\f '(1 1 1 1)))
	append tuple into blist
	sum 4 into blistlen
	finally (return (make-formula :fn 'const
				      :type `(bv ,blistlen)
				      :args blist))))

;;   (let ((h-list (coerce hexval 'list)))
;;     (hex-list-binary (hex-size-error (- (char-code (car h-list)) 48) 
;; 				     (cddr h-list)))))


; for octal
;;;;;;;;;;;;;;;;;;;;;;

; this function converts to binary list of 4 bits exact
(defun bits-3 (blist)
  (if (and (consp blist) 
	   (true-listp blist))
      (if (<= (length blist) 2)
	  (bits-3 (cons '0 blist))
	(if (= (length blist) 3)
	    blist
	  '(0 0 0)))
    '(0 0 0)))

; this function converts a hex-char in string to given bit vector
(defun octal-char-bv (x)
  (let ((ch (char-code x)))
    (if (and (>= ch 48)
	     (<= ch 55))
	(n-ubv (- ch 48))
      nil)))

; this function converts hex list to binary
(defun octal-list-binary (form)
  (if (consp form)
      (let* ((x (car form))
	     (y (octal-char-bv x)))
	(append (bits-3 y) (octal-list-binary (cdr form))))
    nil))  
		    
(defun octal-size (size octal)
  (if (< (length octal) size)
      (octal-size size (append '(#\0) octal))
    octal))
 
(defun octal-size-error (size octal)
  (if (> (length octal) size)
      (format t "Error: Given size is less than required representation size")
    (octal-size size octal)))


;(* main function *)
;;;;;;;;;;;;;;;;;;;;;
(defun octal-binary (octalval)
  (loop for ochar in (cddr (coerce octalval 'list))
	for tuple = (case ochar
		      (#\0 '(0 0 0))
		      (#\1 '(0 0 1))
		      (#\2 '(0 1 0))
		      (#\3 '(0 1 1))
		      (#\4 '(1 0 0))
		      (#\5 '(1 0 1))
		      (#\6 '(1 1 0))
		      (#\7 '(1 1 1)))
	append tuple into blist
	sum 3 into blistlen
	finally (return (make-formula :fn 'const
				      :type `(bv ,blistlen)
				      :args blist))))

;; (defun octal-binary (octalval)
;;   (let ((o-list (coerce octalval 'list)))
;;     (octal-list-binary (octal-size-error (- (char-code (car o-list)) 48) 
;; 					 (cddr o-list)))))

; for binary
;;;;;;;;;;;;;;;;;;;

; this function converts a char in string to given bit vector
(defun char-bv (x)
  (let ((ch (char-code x)))
    (if (or (= ch 48)
	    (= ch 49))
	(n-ubv (- ch 48))
      nil)))

(defun bits (y)
  (if (equal y nil)
      '(0)
    '(1)))

; this function converts hex list to binary
(defun list-binary (form)
  (if (consp form)
      (let* ((x (car form))
	     (y (char-bv x)))
	(append (bits y) (list-binary (cdr form))))
    nil))  
		    
(defun bin-size (size bin)
  (if (< (length bin) size)
      (bin-size size (append '(#\0) bin))
    bin))
 
(defun bin-size-error (size bin)
  (cond ((= size 0) bin)
	((> (length bin) size)
	 (format t "Error: Given size is less than required representation size"))
	(t (bin-size size bin))))

;(* main function *)
;;;;;;;;;;;;;;;;;;;;;;;
(defun fin-binary (binval)
  (let ((b-list (coerce binval 'list)))
    (list-binary (bin-size-error (- (char-code (car b-list)) 48) 
				 (cddr b-list)))))

; for decimal
;;;;;;;;;;;;;;
 
(defun dec-size (size dec)
  (if (< (length dec) size)
      (dec-size size (append '(0) dec))
    dec))
 
(defun dec-size-error (size dec)
  (if (> (length dec) size)
      (format t "Error: Given size is less than required representation size")
    (dec-size size dec)))


;(* main function *) - for the form [0-9]d[0-9]*
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun dec-binary (decval)
  (let ((d-list (coerce decval 'list)))
    (dec-size-error (- (char-code (car d-list)) 48) 
		    (n-ubv (read-from-string (subseq decval 2))))))

;(* main function *) - for the form [0-9]*u
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun un-dec-binary (decval)
  (n-ubv (read-from-string (subseq decval 0 (- (length decval) 1)))))

;(* main function *) - for signed integers
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;		       
(defun s-int-binary (intval)
  (i-sbv (read-from-string intval)))

(defun mv-tc (vars tps form)
  (cond ((endp vars)
	 (or (endp tps)
	     (format t "~&Not enough variables for mv: ~a" form)))
	((endp tps)
	 (format t "~&Too many variables for mv: ~a" form))
	((compare-types (cdar vars) (car tps))
	 (mv-tc (cdr vars) (cdr tps) form))
	(t
	 (format t "~&Incompatible types: ~a: ~a should be type ~a, in ~a"
		 (caar vars) (cdar vars) (car tps) form))))


;; only works properly on non-empty list.
(defun type-check-list (lst tcf)
  (cond ((endp lst) nil)
	((endp (cdr lst))
	 (let ((f (funcall tcf (car lst))))
	   (if f (list f) nil)))
	(t
	 (let ((rst (type-check-list (cdr lst) tcf)))
	   (if rst
	       (let ((f (funcall tcf (car lst))))
		 (if f (cons f rst) nil))
	     nil)))))

;the main type-checking function.
;form is the form being checked.
;db tells us which bits are defined for variables that have been
; defined bit-by-bit
;e is the typing environment
;d is the desc
(defun type-check (form db e d)
;  (format t "~&type-check: ~&form=~a" form)
    (cond ((integerp form)
	   (make-formula :fn 'int
			 :type `(int ,(size form))
			 :args (list form)))
	  ((atom form)
	   (let ((sform (string-downcase (string form))))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; changed by roma on 9 March to add binary nos to const
; changed on 18 may to implement the Verilog representation
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
	   (cond ((and (>= (char-code (elt sform 0)) 48)
		       (<= (char-code (elt sform 0)) 57) 
		       (char-equal (elt sform 1) #\b))
		  (make-formula :fn 'const
				:type `(bv ,(length (fin-binary sform)))
				:args (fin-binary sform)))
		 ((and (>= (char-code (elt sform 0)) 48)
		       (<= (char-code (elt sform 0)) 57) 
		       (char-equal (elt sform 1) #\x))
		  (hex-binary sform))
		 ((and (>= (char-code (elt sform 0)) 48)
		       (<= (char-code (elt sform 0)) 57) 
		       (char-equal (elt sform 1) #\o))
		  (octal-binary sform))
		 ((and (char-equal (elt sform (- (length sform) 1)) #\u)
		       (>= (char-code (elt sform 0)) 48)
		       (<= (char-code (elt sform 0)) 57))
		  (let ((n (read-from-string (string-right-trim '(#\u #\U) sform))))
		    (make-formula :fn 'unsigned
				  :type `(int ,(if (= n 0) 1 (1+ (floor-log n))))
				  :args (list n))))
	         ((not (var-defined? db form e)) 
                  (format t "~&Attempt to use variable that is not fully defined: ~a" form))
                 ((assoc form (desc-defs d))
		  (let* ((def (assoc form e)))
		    (make-formula :fn 'def
				  :type (cdr def)
				  :args (list form))))
                 (t (let* ((def (assoc form e)))
		      (make-formula :fn 'var
				    :type (cdr def)
				    :args (list form)))))))
	  (t (let ((fn (car form))
		   (args (cdr form)))
	       (cond ((isvar fn e)
			(with-formulas
			 ((f (r-type-check fn '(bv int) form db e d)))
			 (let ((tp (formula-type f)))
			   (if (< (first args) (second tp))
			       (if (= (length args) 1)
				   (if (bit-defined? db fn (car args))
				       (make-formula :fn 'bit
						     :args (cons f args))
				     (format t "~&Attempt to use undefined bit: ~a" form))
				 (if (bit-range-defined? db fn (car args) (nth 1 args))
				     (make-formula :fn 'bits
						   :type `(bv ,(1+ (- (nth 1 args) (nth 0 args))))
						   :args (cons f args))
				   (format t "~&Attempt to use undefined bits: ~a" form)))
			     (format t
				     "~&Type error: Index out of bounds: ~a"
				     form)))))
		     ((eq fn '_zero_mem)
		      (make-formula :fn fn
				    :type (cons 'mem args)
				    :args args))
		     ((or (eq fn 'AF)
			  (eq fn 'AG))
		      (let ((fm (type-check (car args) db e d)))
			(when fm
			  (make-formula :fn fn
					:args (list fm)))))
		     ((or (eq fn '<->)
			  (eq fn '->)
			  (eq fn 'and)
			  (eq fn 'or)
			  (eq fn 'xor))
		      (multiple-value-bind
			(tp a)
			(compat-lst args
				    nil
				    (lambda (x) 
				      (r-type-check x '(bv int) form db e d)))
			(when tp
			  (make-formula :fn fn
					:type tp
					:args a))))
		     ((eq fn '<)
		      (multiple-value-bind
			(tp a)
			(compat-lst args
				    nil
				    (lambda (x) 
				      (r-type-check x '(bv int) form db e d)))
			(when tp
			  (make-formula :fn fn
					:args a))))
		     ((eq fn '>)
		      (multiple-value-bind
			(tp a)
			(compat-lst args
				    nil
				    (lambda (x) 
				      (r-type-check x '(bv int) form db e d)))
			(when tp
			  (make-formula :fn '<
					:args (reverse a)))))
		     ((eq fn '<=)
		      (multiple-value-bind
			(tp a)
			(compat-lst args
				    nil
				    (lambda (x) 
				      (r-type-check x '(bv int) form db e d)))
			(when tp
			  (make-formula :fn 'not
					:args (list (make-formula :fn '<
								  :args (reverse a)))))))
		     ((eq fn '>=)
		      (multiple-value-bind
			(tp a)
			(compat-lst args
				    nil
				    (lambda (x) 
				      (r-type-check x '(bv int) form db e d)))
			(when tp
			  (make-formula :fn 'not
					:args (list (make-formula :fn '<
								  :args a))))))
		     ((eq fn '=)
		      (multiple-value-bind
			(tp a)
			(compat-lst args
				    nil
				    (lambda (x)
				      (type-check x db e d)))
			(when tp
			  (make-formula :fn fn
					:args a))))
		     ((or (eq fn '+)
			  (eq fn '-))
		      (multiple-value-bind
			(tp a)
			(compat-lst args
				    nil
				    (lambda (x) 
				      (r-type-check x '(bv int) form db e d)))
			(when tp
			  (make-formula :fn fn
					:type (list (first tp)
						    (+ (second tp) (ceil-log (length a))))
					:args a))))
		     ((or (eq fn 'add)
			  (eq fn 'sub)
			  (eq fn 'neg)
			  (eq fn 'inc)
			  (eq fn 'dec)
			  (eq fn 'mult))
		      (multiple-value-bind
			(tp a)
			(compat-lst args
				    nil
				    (lambda (x) 
				      (r-type-check x '(bv int) form db e d)))
			(when tp
			  (make-formula :fn fn
					:type (list (first tp)
						    (+ (second tp) 1))
					:args a))))
		     ((or (eq fn 'mod+)
			  (eq fn 'mod-)
			  (eq fn 'mod*))
		      (multiple-value-bind
			(tp a)
			(compat-lst args
				    nil
				    (lambda (x) 
				      (r-type-check x '(bv int) form db e d)))
			(when tp
			  (make-formula :fn fn
					:type (list (first tp)
						    (second tp))
					:args a))))
		     ((eq fn 'cond)
		      (let ((conds (mapcar #'car args))
			    (exps (mapcar #'cadr args))
			    (args nil))
			(multiple-value-bind
			  (tp a)
			  (compat-lst exps
				      nil
				      (lambda (x)
					(type-check x db e d)))
			  (when tp
			    (dolist (c conds (if (or (eq (car tp) 'mem)
						     (eq (car tp) 'mv))
						 (format t "~&Error: must give default case for conds with memory or mv type.~% You wrote ~A~%" form)
					       (make-formula :fn fn
							     :type tp
							     :args (reverse args))))
			      (let ((tc (type-check-1 c db e d)))
				(cond ((not tc) (return nil))
				      ((and (eq (formula-fn tc) 'const)
					    (eq (car (formula-args tc)) 1))
				       (return (make-formula :fn fn
							     :type tp
							     :args (reverse (cons (list tc (car a)) args)))))
				      (t (setf args (cons (list tc (car a)) args))
					 (setf a (cdr a))))))))))
		     ((eq fn 'local)
		      (let* ((len (length args))
			     (a1 (if (= len 3) (fix-var-list (nth 0 args) d) nil))
			     (vars (mapcar #'car a1))
			     (a2 (if (= len 3) (second args) (first args)))
			     (a3 (if (= len 3) (third args) (second args))))
			(setf e (append a1 e))
			(dolist (a a1 t)
			  (setf db (cons (list (car a)) db)))
			(multiple-value-bind
			  (db e bforms)
			  (local-binding-type-check a2 vars db e d)
			  (when e
			    (let ((tc (type-check a3 db e d)))
			      (when tc
				(make-formula :fn 'local
					      :type (formula-type tc)
					      :args (list a1 bforms tc))))))))
		     ((eq fn 'if)
		      (let ((testf (type-check-1 (first args) db e d)))
			(when testf
			  (multiple-value-bind
			    (tp a)
			    (compat-lst (cdr args)
					nil
					(lambda (x)
					  (type-check x db e d)))
			    (when tp
			      (make-formula :fn 'if
					    :type tp
					    :args (cons testf a)))))))
;; 			(when testf
;; 			  (let ((thenf (type-check (second args) db e d)))
;; 			    (when thenf
;; 			      (let ((elsef (type-check (third args) db e d)))
;; 				(when elsef
;; 				  (if (compatible-bitsizes (formula-bits thenf)
;; 							   (formula-bits elsef))
;; 				      (progn
;; 					(setf wordsize-if (pick-wordsize 
;; 							   (formula-wordsize thenf)
;; 							   (formula-wordsize elsef)))
;; 					(make-formula :fn fn
;; 						      :bits (pick-bitsize (formula-bits thenf)
;; 									  (formula-bits elsef))
;; 						      :wordsize wordsize-if
;; 						      :args (list testf thenf elsef)))
;; 				    (format t 
;; 					    "~&Type Error: Incompatible then and else clauses: ~&~a"
;; 					      form)))))))))
		     ((eq fn 'mv)
		      (let ((lst (type-check-list args 
						  (lambda (x)
						    (r-type-check x '(mem bv int) form db e d)))))
			(if lst
			    (make-formula :fn 'mv
					  :type (cons 'mv 
						      (mapcar #'formula-type lst))
					  :args lst))))
		     ((eq fn 'mv-let)
		      (let ((vars (fix-var-list (first args) d))
			    (mv (type-check (second args) db e d)))
			(when mv
			  (if (eq (first (formula-type mv)) 'mv)
			      (when (mv-tc vars (cdr (formula-type mv)) form)
				(let ((body (type-check (third args)
							db
							(append vars e)
							d)))
				  (when body
				    (make-formula :fn 'mv-let
						  :type (formula-type body)
						  :args (list vars mv body)))))
			    (format t "~&Attempting mv-let on single-valued form: ~a" form)))))
		     ((eq fn 'cat)
		      (loop for x in args
			    for curform = (r-tl-type-check x '(bv int) form db e d)
			    unless curform return nil
			    summing (second (formula-type curform)) into sum
			    collecting curform into newargs
			    finally (return (make-formula :fn fn
							  :type `(bv ,sum)
							  :args newargs))))
		     ((eq fn '*)
		      (loop for x in args
			    for curform = (r-type-check x '(bv int) form db e d)
			    unless curform return nil
			    summing (second (formula-type curform)) into sum
			    collecting curform into newargs
			    finally (return (make-formula :fn fn
							  :type `(bv ,sum)
							  :args newargs))))
;; 		     ((eq fn '->)
;; 		      (multiple-value-bind
;; 			(tp a)
;; 			(compat-lst args
;; 				    nil
;; 				    (lambda (x) 
;; 				      (r-type-check x '(bv int) form db e d)))
		      
;; 		      (let ((f0 (type-check (first args) db e d)))
;; 			(when f0
;; 			  (let ((f1 (type-check (second args) db e d)))
;; 			    (when f1
;; 			      (make-formula :fn fn
;; 					    :args (list f0 f1)))))))
		     ((or (eq fn '>>)
			  (eq fn '<<)
			  (eq fn '>>>)
			  (eq fn '<<<))
		      (let ((f0 (r-type-check (first args) '(bv int) form db e d)))
			(when f0
			  (make-formula :fn fn
					:type (formula-type f0)
					:args (list f0 (nth 1 args))))))
		     ((eq fn 'next)
		      (let ((f (type-check (first args) db e d)))
			(when f
			  (make-formula :fn fn
					:type (formula-type f)
					:args (list f)))))
		     ((or (eq fn 'not))
		      (let ((f (r-type-check (first args) '(bv int) form db e d)))
			(when f
			  (make-formula :fn fn
					:type (formula-type f)
					:args (list f)))))
		     ((or (eq fn 'foldl)
			  (eq fn 'foldr))
		      (let* ((op (car args))
			     (exp (second args))
			     (funct (get-funct op d)))
			(if (and funct
				 (not (and (= (length (funct-params funct)) 2)
					   (dolist (p (funct-params funct) t)
					     (when (not (= (nth 1 p) 1)) (return nil))))))
			    (format t 
				    "~&Type Error: Function passed to ~a must be a binary function with parameters of type 1. You wrote ~a"
				    fn
				    form)
			  (let ((f (r-type-check exp '(bv int) form db e d)))
			    (when f
			      (make-formula :fn fn
					    :args (list op f)))))))
		     ((eq fn 'type)
		      (let ((f (r-type-check (first args) '(bv int) form db e d)))
			(when f
			  (make-formula :fn fn
					:type (list 'bv (second args))
					:args (list f)))))
		     ((eq fn 'bit)
		      (let ((f (r-tl-type-check (nth 0 args) '(bv int) form db e d)))
			(when f
			  (if (< (nth 1 args)
				 (second (formula-type f)))
			      (make-formula :fn fn
					    :args (cons f (cdr args)))
			    (format t
				    "~&Type Error: Bit number out of bounds: ~a"
				    form)))))
		     ((eq fn 'bits)
		      (let ((f (r-tl-type-check (nth 0 args) '(bv int) form db e d)))
			(when f
			  (if (< (nth 2 args)
				 (second (formula-type f)))
			      (make-formula :fn fn
					    :type `(bv ,(1+ (- (nth 2 args)
							      (nth 1 args))))			    
					    :args (cons f (cdr args)))
			    (format t
				    "~&Type Error: Bit number out of bounds: ~a"
				    form)))))
;; changed by roma on Nov 23,05 to type-check get and set functions
 		     ((eq fn 'get)
		      (with-formulas 
		       ((f1 (r-type-check (first args) '(mem) form db e d))
			(f2 (r-type-check (second args) '(bv int) form db e d)))
		       (let ((f3 (third args)))
			 (let* ((nw (second (formula-type f1)))
				(ws (third (formula-type f1)))
				(lnw (ceil-log nw))
				(ct (compare-types `(bv ,lnw) (formula-type f2))))
			   (cond ((not (or (eq ct '<=) (eq ct '=)))
				  (format t "~& Address value is out of the wrong type: ~a of type ~a should be of type ~a."
					  (second args) (formula-type f2) lnw))
				 (t
				  (make-formula :fn 'get
						:type `(bv ,(if (= (length args) 3)
								(* ws f3)
							      ws))
						:args (cons f1 (cons f2 (list (if f3 f3 1)))))))))))
		     ((eq fn 'set)
		      (with-formulas 
		       ((f1 (r-type-check (first args) '(mem) form db e d))
			(f2 (r-type-check (second args) '(bv int) form db e d))
			(f3 (r-type-check (third args) '(bv int) form db e d)))
		       (let* ((nw (second (formula-type f1)))
			      (lnw (ceil-log nw))
			      (ct (compare-types `(bv ,lnw) (formula-type f2))))
			 (cond ((not (or (eq ct '<=) (eq ct '=)))
				(format t "~& Address is of the wrong type:~%   ~a of type ~a should be of type ~a.~%"
					(second args) (formula-type f2) `(bv ,lnw)))
			       ((let ((ct2 (compare-types `(bv ,(third (formula-type f1))) (formula-type f3))))
				  (not (or (eq ct2 '<=) (eq ct2 '=))))
				(format t "~& Value in set statement is of the wrong type:~%   ~A of type ~A should be of type ~A.~%"
					(third args) (formula-type f3) `(bv ,(third (formula-type f1)))))
			       (t
				(make-formula :fn 'set
					      :type (formula-type f1)
					      :args (list f1 f2 f3)))))))
		     ((eq fn 'ext)
		      (let ((f (r-type-check (first args) '(bv int) form db e d)))
			(when f
			  (if (< (second (formula-type f))
				 (second args))
			      (make-formula :fn fn
					    :type `(bv ,(second args))
					    :args (list f))
			    (format t
				    "~&Type Error: Extension type too small: ~a"
				    form)))))
		     ((eq fn 'const)
		      (make-formula :fn fn
				    :type `(bv ,(length args))
				    :args args))
		     (t
		      (let ((f (get-funct fn d)))
			(when f
			  (let* ((type (funct-type f))
				 (params (funct-params f))
				 (len (length params))
				 (len2 (length args))
				 (nargs nil))
			    (if (= len len2)
				(dolist (p params (make-formula :fn fn
								:type type
								:args (reverse nargs)))
				  (let* ((ptype (cdr p))
					 (aform (type-check (first args) db e d)))
				    (if aform
					(let ((ct (compare-types ptype (formula-type aform))))
					  (if (or (eq ct '<=) (eq ct '=))
					      (setf nargs (cons aform nargs))
					    (return (format t
							    "~&Type error: Type mismatch: Form: ~A: Attempt to pass ~a of type ~a as type ~a"
							    form
							    (first args)
							    (formula-type aform)
							    ptype))))
				      (return nil)))
				  (setf args (cdr args)))
			      (format t
				      "~&Type error: ~a expects ~a arguments. You gave it ~a: ~&~a"
				      fn
				      len
				      len2
				      form)))))))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; int fixing                        ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;fixes a list of formulas.
;lst is the list being fixed
;bits is the bits to give the elements of the list
;d is the desc
(defun fix-ints-list (lst bits d)
  (let ((f #'(lambda (form) (fix-ints form bits d))))
    (mapcar f lst)))

;top-level fixes ints in a list.
(defun tl-fix-ints-list (lst d)
  (let ((f #'(lambda (x) (tl-fix-ints x d))))
    (mapcar f lst)))

;finds the most restrictive bits in a list. this is the bits we will
;give everything in the list.
;; (defun list-bits (lst)
;;   (let ((tp 0))
;;     (dolist (x lst tp)
;;       (let ((xtp (formula-bits x)))
;; 	(cond ((not (eq (formula-type x) 'int))
;; 	       (return xtp))
;; 	      ((>= xtp tp)
;; 	       (setf tp xtp)))))))

;; 	(if (atom xtp)
;; 	    (return (setf tp xtp))
;; 	  (when
;; 	  (if (>= (minbitsizextp) (minbitsize tp))
;; 	      (setf tp xtp)))))
;;     (minbitsize tp)))

;converts an integer into the corresponding signed bit-vector of the given bits.
(defun make-int-bits (i bits)
  (let ((s (size i)))
    (if (< bits s)
	(nthcdr (- s bits) (i-sbv i))
      (sign-extend (i-sbv i) bits))))

(defun make-uint-bits (i bits)
  (let ((s (usize i)))
    (if (< bits s)
	(nthcdr (- s bits) (n-ubv i))
      (extend (n-ubv i) bits))))

(defun fix-type (tp)
  (case (car tp)
    (int `(bv ,(second tp)))
    (mv (cons 'mv (mapcar #'fix-type (cdr tp))))
    (otherwise tp)))
     
;top-level fixing. it chooses the most restrictive bits possible for
;form.
(defun tl-fix-ints (form d)
  (fix-ints form (fix-type (formula-type form)) d))

;the main fixign algorithm.
(defun fix-ints (form tp d)
  (cond ((not (formula-p form))
	 form)
	((not (typep tp 'form-type))
	 (break (format nil "fix-ints: improper type: ~A" tp)))
	(t
	 (setf (formula-type form) tp)
	 ;; (format t "~&fix-ints. form=~a" form)
	 (let ((fn (formula-fn form))
	       (args (formula-args form)))
	   (cond ((member fn '(int type) :test 'eq)
		  (setf (formula-fn form) 'const)
		  (setf (formula-args form) (make-int-bits (car (formula-args form)) (second tp))))
		 ((eq fn 'unsigned)
		  (setf (formula-fn form) 'const)
		  (setf (formula-args form) (make-uint-bits (car (formula-args form)) (second tp))))
		 ((or (eq fn 'AG)
		      (eq fn 'AF)
		      (eq fn 'cat)
		      (eq fn '*))
		  (tl-fix-ints-list args d))
		 ((member fn '(<-> and or xor not inc dec next -> mod+ mod- mod*) :test 'eq)
		  (fix-ints-list args tp d))
		 ((or (eq fn 'foldl)
		      (eq fn 'foldr))
		  (tl-fix-ints (second args) d))
		 ((eq fn 'local)
		  (let ((bindings (nth 1 args)))
		    (fix-ints (third args) tp d)
		    (dolist (b bindings)
		      (let ((btp (formula-type b))
			    (bargs (formula-args b)))
			(fix-ints (car (last bargs)) btp d)))))
		 ((or (eq fn '+)
		      (eq fn '-))
		  (fix-ints-list args 
				 `(bv ,(- (second tp) (ceil-log (length args))))
				 d))
		 ((or (eq fn 'add)
		      (eq fn 'sub)
		      (eq fn 'neg)
		      (eq fn 'inc)
		      (eq fn 'dec)
		      (eq fn 'mult))
		  (fix-ints-list args
				 `(bv ,(- (second tp) 1))
				 d))
		 ((or (eq fn '=)
		      (eq fn '<))
		  (fix-ints-list args
				 (fix-type (most-restrictive-type 
					    (formula-type (first args)) 
					    (formula-type (second args))))
				 d))
		 ((eq fn 'mv)
		  (mapcar (lambda (x type) (fix-ints x type d))
			  args
			  (cdr tp)))
		 ((eq fn 'mv-let)
		  (fix-ints (second args) (cons 'mv (mapcar #'cdr (first args))) d)
		  (fix-ints (third args) tp d))		  
		 ((eq fn 'cond)
		  (mapcar (lambda (x)
			    (tl-fix-ints (first x) d)
			    (fix-ints (second x) tp d))
			  args))
		 ((eq fn 'if)
		  (tl-fix-ints (car args) d)
		  (fix-ints-list (cdr args) tp d))
		 ((or (eq fn '>>)
		      (eq fn '<<))
		  (fix-ints (nth 0 args) tp d))
		 ((eq fn 'get)
		  (let ((mtp (formula-type (first args))))
		    (tl-fix-ints (first args) d)
		    (fix-ints (second args) `(bv ,(ceil-log (second mtp))) d)))
		 ((eq fn 'set)
		  (fix-ints (first args) tp d)
		  (fix-ints (second args) `(bv ,(ceil-log (second tp))) d)
		  (fix-ints (third args) `(bv ,(third tp)) d))
		 ((or (eq fn 'bit)
		      (eq fn 'bits))
		  (let ((mb (max (car (last args)) (second (formula-type (first args))))))
		    (fix-ints (first args) `(bv ,mb) d)))
		 ((eq fn 'ext)
		  (tl-fix-ints (car args) d))
		 ((get-funct fn d)
		  (let* ((f (get-funct fn d))
			 (params (funct-params f)))
		    (dolist (p params)
		      (fix-ints (car args) (cdr p) d)
		      (setf args (cdr args)))))
		 (t form)))
	 form)))
  
;fix a function funct in desc d.
(defun fix-function (funct d)
  (let* ((fn (cdr funct))
	 (tcf (tl-type-check (replace-constants (funct-body fn) d)
			     nil
			     (funct-params fn)
			     d)))
    (when tcf
      (cond ((subtypeof (funct-type fn) (formula-type tcf))
	     (setf (funct-body fn)
		   (fix-ints tcf (funct-type fn) d)))
	    (t (format t "~&Function ~a: function type ~a does not match body type: ~a"
		       (car funct)
		       (funct-type fn)
		       (formula-type tcf)))))))

;fix the defs section in d.
(defun fix-defs (d)
  (let ((defs (desc-defs d))
        (ndefs nil)
        (e (desc-vars d)))
    (dolist (def defs (progn (setf (desc-defs d) (reverse ndefs)) t))
      (let ((ndef (tl-type-check (replace-constants (second def) d)
                                 nil
                                 e
                                 d)))
        (cond (ndef
               (tl-fix-ints ndef d)
               (setf ndefs (acons (first def) ndef ndefs))
               (setf e (acons (first def) (formula-type ndef) e)))
              (t (return nil)))))))

;auxilary function for desc-tl-env
(defun desc-tl-env-aux (defs env)
  (if (consp defs)
      (desc-tl-env-aux (cdr defs) 
                       (acons (caar defs) (formula-type (cdar defs)) env))
    env))

;turns the vars and defs of d into a typing environment.
(defun desc-tl-env (d)
  (desc-tl-env-aux (reverse (desc-defs d)) (desc-vars d)))

;fixes the init, trans, or spec section of d as specified.
(defun fix-section (d sect)
  (let ((e (desc-tl-env d)))
    (when (case sect
	    (init  (let ((s (type-check-1 (replace-constants (desc-init d) d) nil e d)))
		     (when s 
		       (setf (desc-init d) (tl-fix-ints s d)))))
	    (trans (let ((s (type-check-1 (replace-constants (desc-trans d) d) nil e d)))
		     (when s 
		       (setf (desc-trans d) (tl-fix-ints s d)))))
	    (spec  (let ((s (type-check-1 (replace-constants (desc-spec d) d) nil e d)))
		     (when s 
		       (setf (desc-spec d) (tl-fix-ints s d))))))
      t)))

;; function for counting the number of reads and writes to a particular memory
;; added: roma, 12 Dec05
;; (defun mem-count (form assoc-list)
;; ;  (format t "~&mem-count. form: ~a" form)
;;   (let ((fn (car form))
;; 	(args (cdr form)))
;;     (cond ((eq fn 'if)
;; 	   (let ((then-assoc (mem-count (second args) assoc-list))
;; 		 (else-assoc (mem-count (third-args) assoc-list)))
;; 	     (append then-assoc else-assoc)))
;; 	  ((eq fn 'get)
;; 	   (setf assoc-list 
;; 		 (incf (assoc (first args) assoc-list))))
;; 	  ((eq fn 'set)
;; 	   (setf assoc-list 
;; 		 (incf (assoc (first args) assoc-list))))
	  
	       

